// Runs in ISOLATED world — relays intercepted API data + scrapes DOM on /settings
(function () {
  'use strict';

  // ── Guard: stop everything if extension context is dead ─────────────
  let contextValid = true;
  let refreshInterval = null;

  function safeSendMessage(data) {
    if (!contextValid) return;
    try {
      chrome.runtime.sendMessage(data);
    } catch (e) {
      if (e.message.includes('Extension context invalidated')) {
        contextValid = false;
        cleanup();
      }
    }
  }

  function cleanup() {
    if (refreshInterval) clearInterval(refreshInterval);
    if (observer) observer.disconnect();
  }

  // ── Relay messages from MAIN-world interceptor ────────────────────────
  window.addEventListener('message', (event) => {
    if (event.source !== window) return;
    const { type } = event.data;
    if (
      type === 'CLAUDE_LIMITS_API_RESPONSE' ||
      type === 'CLAUDE_LIMITS_RATE_HEADERS'
    ) {
      safeSendMessage(event.data);
    }
  });

  // ── DOM scraping (only on settings page) ──────────────────────────────
  if (!window.location.pathname.includes('/settings')) return;

  function scrapeUsageData() {
    if (!contextValid) return;

    const data = {
      type: 'CLAUDE_LIMITS_DOM_SCRAPE',
      timestamp: Date.now(),
      url: window.location.href,
      limits: [],
      resetInfo: [],
      rawSections: [],
    };

    const body = document.body;
    if (!body) return;

    // Grab full page text for raw section fallback
    const pageText = body.innerText || '';

    // ── Primary: find "X% used" patterns and associate with labels ──────
    const allTextEls = [];

    // Walk all elements looking for "X% used" text
    const walker = document.createTreeWalker(body, NodeFilter.SHOW_TEXT, null);
    while (walker.nextNode()) {
      const node = walker.currentNode;
      const text = node.textContent.trim();
      if (text.length > 0) {
        allTextEls.push({ node, text });
      }
    }

    // Find elements containing "X% used"
    for (const { node, text } of allTextEls) {
      const match = text.match(/([\d.]+)%\s*used/i);
      if (!match) continue;

      const pct = parseFloat(match[1]);
      const sectionInfo = findSectionInfo(node);

      data.limits.push({
        type: classifyLabel(sectionInfo.label),
        label: sectionInfo.label,
        percentage: pct,
        resetAt: sectionInfo.resetTime,
        source: 'text_used',
      });
    }

    // ── Fallback: parse page text line-by-line if primary found nothing ──
    if (data.limits.length === 0) {
      const lines = pageText.split('\n').map(l => l.trim()).filter(Boolean);
      let currentLabel = '';
      let currentReset = '';

      for (const line of lines) {
        // Detect section labels
        if (/current\s+session/i.test(line)) { currentLabel = 'session'; currentReset = ''; }
        else if (/all\s+models?/i.test(line) && !/turn/i.test(line)) { currentLabel = 'total'; currentReset = ''; }
        else if (/sonnet\s+only/i.test(line)) { currentLabel = 'model_sonnet'; currentReset = ''; }
        else if (/opus\s+only/i.test(line)) { currentLabel = 'model_opus'; currentReset = ''; }
        else if (/extra\s+usage/i.test(line)) { currentLabel = 'extra'; currentReset = ''; }

        // Detect reset times
        const resetMatch = line.match(/resets?\s+(?:in\s+)?(.+)/i);
        if (resetMatch) {
          currentReset = resetMatch[1].trim();
        }

        // Detect "X% used"
        const pctMatch = line.match(/([\d.]+)%\s*used/i);
        if (pctMatch && currentLabel) {
          data.limits.push({
            type: currentLabel,
            label: currentLabel,
            percentage: parseFloat(pctMatch[1]),
            resetAt: currentReset,
            source: 'text_fallback',
          });
        }
      }
    }

    // ── Collect reset info separately ───────────────────────────────────
    for (const { node, text } of allTextEls) {
      const resetMatch = text.match(/resets?\s+(?:in\s+)?(.+)/i);
      if (resetMatch) {
        const sectionInfo = findSectionInfo(node);
        data.resetInfo.push({
          text: text,
          timeString: resetMatch[1].trim(),
          label: sectionInfo.label,
          type: classifyLabel(sectionInfo.label),
        });
      }
    }

    // ── Collect extra usage / spend info ────────────────────────────────
    for (const { text } of allTextEls) {
      const spendMatch = text.match(/[€$£]([\d,.]+)\s*spent/i);
      if (spendMatch) {
        data.extraSpend = spendMatch[0];
      }
    }

    // ── Raw section text for debug ──────────────────────────────────────
    const sections = pageText.match(/Plan usage limits[\s\S]*?(?=Extra usage|$)/i);
    if (sections) {
      data.rawSections.push(sections[0].substring(0, 1500));
    }
    const extraSection = pageText.match(/Extra usage[\s\S]*?(?=\n\n|$)/i);
    if (extraSection) {
      data.rawSections.push(extraSection[0].substring(0, 500));
    }

    if (data.limits.length > 0 || data.resetInfo.length > 0) {
      safeSendMessage(data);
    }
  }

  // ── Walk up the DOM from a text node to find section label + reset ────
  function findSectionInfo(textNode) {
    let label = '';
    let resetTime = '';

    let container = textNode.parentElement;
    for (let depth = 0; depth < 10 && container; depth++) {
      const innerText = container.innerText || '';
      const lines = innerText.split('\n').map(l => l.trim()).filter(Boolean);

      for (const line of lines) {
        if (!label) {
          if (/current\s+session/i.test(line)) label = 'Current session';
          else if (/all\s+models?$/i.test(line)) label = 'All models';
          else if (/sonnet\s+only/i.test(line)) label = 'Sonnet only';
          else if (/opus\s+only/i.test(line)) label = 'Opus only';
          else if (/haiku\s+only/i.test(line)) label = 'Haiku only';
          else if (/extra\s+usage/i.test(line)) label = 'Extra usage';
          else if (/weekly\s+limits?/i.test(line)) label = 'Weekly limits';
        }
        if (!resetTime) {
          const rm = line.match(/resets?\s+(?:in\s+)?(.+)/i);
          if (rm) resetTime = rm[1].trim();
        }
      }

      if (label) break;
      container = container.parentElement;
    }

    return { label: label || 'unknown', resetTime };
  }

  // ── Classify label into our metric keys ───────────────────────────────
  function classifyLabel(label) {
    if (!label) return 'unknown';
    const t = label.toLowerCase();
    if (t.includes('session')) return 'session';
    if (t.includes('all model') || t.includes('total')) return 'total';
    if (t.includes('sonnet')) return 'model_sonnet';
    if (t.includes('opus')) return 'model_opus';
    if (t.includes('haiku')) return 'model_haiku';
    if (t.includes('extra')) return 'extra';
    if (t.includes('weekly')) return 'weekly';
    if (t.includes('daily')) return 'daily';
    if (t.includes('monthly')) return 'monthly';
    return 'unknown';
  }

  // ── Run scraper with retry + MutationObserver ─────────────────────────
  let scrapeTimer;
  function scheduleScrape() {
    if (!contextValid) return;
    clearTimeout(scrapeTimer);
    scrapeTimer = setTimeout(scrapeUsageData, 1500);
  }

  // Initial scrapes (page may load progressively)
  setTimeout(scrapeUsageData, 2000);
  setTimeout(scrapeUsageData, 5000);
  setTimeout(scrapeUsageData, 10000);

  const observer = new MutationObserver(scheduleScrape);
  observer.observe(document.body, { childList: true, subtree: true, characterData: true });

  // ── Periodic auto-refresh every 90 seconds ─────────────────────────────
  refreshInterval = setInterval(() => {
    if (!contextValid) return;
    // Try to click "Last updated" to make Claude.ai fetch fresh server data
    const allEls = document.querySelectorAll('button, a, [role="button"], span');
    for (const el of allEls) {
      const text = (el.textContent || '').trim().toLowerCase();
      if (text.includes('last updated') || text.includes('refresh') || text === 'update') {
        el.click();
        break;
      }
    }
    // Re-scrape after a short delay to let any triggered update render
    setTimeout(scrapeUsageData, 2000);
  }, 90 * 1000);

  // ── Listen for force-scrape requests from background ────────────────────
  chrome.runtime.onMessage.addListener((msg) => {
    if (!contextValid) return;
    if (msg.type === 'FORCE_SCRAPE') {
      scrapeUsageData();
    }
  });
})();
