// Runs in MAIN world to intercept fetch/XHR responses from claude.ai
(function () {
  'use strict';

  const MARKER = '__CLAUDE_LIMITS_INTERCEPTOR__';
  if (window[MARKER]) return;
  window[MARKER] = true;

  // --- Fetch interception ---
  const originalFetch = window.fetch;
  window.fetch = async function (...args) {
    const response = await originalFetch.apply(this, args);

    try {
      const url = typeof args[0] === 'string' ? args[0] : args[0]?.url || '';

      if (shouldCapture(url)) {
        const clone = response.clone();
        clone.json().then(data => {
          window.postMessage({
            type: 'CLAUDE_LIMITS_API_RESPONSE',
            url,
            status: response.status,
            data,
          }, '*');
        }).catch(() => { });
      }

      // Capture rate-limit headers from any claude.ai API response
      if (url.includes('/api/') || url.includes('claude.ai')) {
        const rlHeaders = extractRateLimitHeaders(response);
        if (rlHeaders) {
          window.postMessage({
            type: 'CLAUDE_LIMITS_RATE_HEADERS',
            url,
            headers: rlHeaders,
          }, '*');
        }
      }
    } catch (e) { /* swallow */ }

    return response;
  };

  // --- XHR interception ---
  const origOpen = XMLHttpRequest.prototype.open;
  const origSend = XMLHttpRequest.prototype.send;

  XMLHttpRequest.prototype.open = function (method, url, ...rest) {
    this._clmUrl = url;
    return origOpen.apply(this, [method, url, ...rest]);
  };

  XMLHttpRequest.prototype.send = function (...args) {
    this.addEventListener('load', function () {
      try {
        if (shouldCapture(this._clmUrl)) {
          const data = JSON.parse(this.responseText);
          window.postMessage({
            type: 'CLAUDE_LIMITS_API_RESPONSE',
            url: this._clmUrl,
            status: this.status,
            data,
          }, '*');
        }
      } catch (e) { /* swallow */ }
    });
    return origSend.apply(this, args);
  };

  function shouldCapture(url) {
    if (!url) return false;
    const patterns = [
      '/usage', '/rate_limit', '/billing', '/subscription',
      '/quota', '/settings', '/account', '/organization',
    ];
    return patterns.some(p => url.includes(p));
  }

  function extractRateLimitHeaders(response) {
    const headers = {};
    const prefixes = ['x-ratelimit', 'retry-after', 'x-usage', 'x-rate'];
    for (const [key, value] of response.headers.entries()) {
      if (prefixes.some(p => key.toLowerCase().startsWith(p))) {
        headers[key] = value;
      }
    }
    return Object.keys(headers).length > 0 ? headers : null;
  }
})();
