(function () {
  'use strict';

  const container = document.getElementById('limitsContainer');
  const emptyState = document.getElementById('emptyState');
  const lastUpdatedEl = document.getElementById('lastUpdated');
  const bridgeDot = document.getElementById('bridgeDot');
  const bridgeLabel = document.getElementById('bridgeLabel');
  const rawDebug = document.getElementById('rawDebug');

  const DISPLAY_KEYS = [
    'session', 'daily', 'weekly', 'monthly', 'total',
    'model_opus', 'model_sonnet', 'model_haiku', 'extra', 'rateLimit',
  ];

  const LABELS = {
    session: 'Session (5h)',
    daily: 'Daily',
    weekly: 'Weekly',
    monthly: 'Monthly',
    total: 'All Models',
    model_opus: 'Opus',
    model_sonnet: 'Sonnet',
    model_haiku: 'Haiku',
    extra: 'Extra Usage',
    rateLimit: 'Rate Limit',
  };

  function pctColor(pct) {
    if (pct >= 90) return 'red';
    if (pct >= 70) return 'yellow';
    return 'green';
  }

  function timeAgo(ts) {
    if (!ts) return '—';
    const diff = Math.floor((Date.now() - ts) / 1000);
    if (diff < 60) return `${diff}s ago`;
    if (diff < 3600) return `${Math.floor(diff / 60)}m ago`;
    if (diff < 86400) return `${Math.floor(diff / 3600)}h ago`;
    return `${Math.floor(diff / 86400)}d ago`;
  }

  function renderLimits(data) {
    container.innerHTML = '';
    let hasLimits = false;

    for (const key of DISPLAY_KEYS) {
      const item = data[key];
      if (!item || item.percentage === undefined || item.percentage === null) continue;
      hasLimits = true;

      const pct = Math.min(100, Math.max(0, item.percentage));
      const color = pctColor(pct);
      const label = item.label && !LABELS[key] ? item.label : (LABELS[key] || key);

      const card = document.createElement('div');
      card.className = 'limit-card';
      card.innerHTML = `
        <div class="limit-header">
          <span class="limit-label">${label}</span>
          <span class="limit-pct ${color}">${pct.toFixed(1)}%</span>
        </div>
        <div class="progress-track">
          <div class="progress-fill ${color}" style="width: ${pct}%"></div>
        </div>
        <div class="limit-meta">
          <span>${item.used !== undefined && item.total !== undefined ? `${item.used} / ${item.total}` : (item.source || '')}</span>
          <span>${item.resetAt ? `Resets: ${item.resetAt}` : (item.resetIn || '')}</span>
        </div>
      `;
      container.appendChild(card);
    }

    // Also show any unknown scraped limits
    for (const [key, item] of Object.entries(data)) {
      if (DISPLAY_KEYS.includes(key) || typeof item !== 'object' || item === null) continue;
      if (key.startsWith('scraped_') && item.percentage !== undefined) {
        hasLimits = true;
        const pct = Math.min(100, Math.max(0, item.percentage));
        const color = pctColor(pct);

        const card = document.createElement('div');
        card.className = 'limit-card';
        card.innerHTML = `
          <div class="limit-header">
            <span class="limit-label">${item.label || key}</span>
            <span class="limit-pct ${color}">${pct.toFixed(1)}%</span>
          </div>
          <div class="progress-track">
            <div class="progress-fill ${color}" style="width: ${pct}%"></div>
          </div>
        `;
        container.appendChild(card);
      }
    }

    // Show reset info if available
    if (data.resetInfo?.length > 0) {
      for (const ri of data.resetInfo) {
        const el = document.createElement('div');
        el.className = 'limit-card';
        el.innerHTML = `<div class="limit-label" style="font-size:11px; color:#9ca3af">${ri.text}</div>`;
        container.appendChild(el);
      }
    }

    emptyState.style.display = hasLimits ? 'none' : 'block';
    container.style.display = hasLimits ? 'flex' : 'none';
    lastUpdatedEl.textContent = data.lastUpdated ? `Updated ${timeAgo(data.lastUpdated)}` : '—';

    // Raw debug
    const debugData = { ...data };
    delete debugData.rawSections; // Too noisy
    rawDebug.textContent = JSON.stringify(debugData, null, 2);
  }

  function loadData() {
    chrome.runtime.sendMessage({ type: 'GET_USAGE_DATA' }, (data) => {
      if (chrome.runtime.lastError) {
        emptyState.style.display = 'block';
        container.style.display = 'none';
        return;
      }
      renderLimits(data || {});
    });

    chrome.runtime.sendMessage({ type: 'GET_BRIDGE_STATUS' }, (status) => {
      if (chrome.runtime.lastError) return;
      if (status?.connected) {
        bridgeDot.className = 'status-dot connected';
        bridgeLabel.textContent = 'bridge connected';
      } else {
        bridgeDot.className = 'status-dot warning';
        bridgeLabel.textContent = 'bridge offline';
      }
    });
  }

  document.getElementById('btnRefresh').addEventListener('click', loadData);
  document.getElementById('btnDebug').addEventListener('click', () => {
    rawDebug.classList.toggle('visible');
  });

  loadData();
})();
